<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dima M3ak - Suivi de Localisation en Direct</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            color: #333;
        }

        .header {
            background: rgba(255, 255, 255, 0.15);
            backdrop-filter: blur(20px);
            padding: 2rem;
            color: white;
            text-align: center;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.2);
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .header h1 {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
        }

        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
            font-weight: 300;
        }

        .container {
            flex: 1;
            display: flex;
            gap: 2rem;
            padding: 2rem;
            max-width: 1600px;
            margin: 0 auto;
            width: 100%;
        }

        .sidebar {
            width: 350px;
            background: rgba(255, 255, 255, 0.98);
            border-radius: 20px;
            padding: 2rem;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
            backdrop-filter: blur(20px);
            height: fit-content;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .map-container {
            flex: 1;
            background: rgba(255, 255, 255, 0.98);
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        #map {
            width: 100%;
            height: 700px;
            border-radius: 20px;
        }

        .status {
            padding: 1.5rem;
            border-radius: 16px;
            margin-bottom: 2rem;
            font-weight: 600;
            font-size: 1rem;
            text-align: center;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }

        .status.active {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            color: #155724;
            border: 2px solid #b8dacc;
        }

        .status.ended {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
            border: 2px solid #f1b0b7;
        }

        .status.error {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
            border: 2px solid #f1b0b7;
        }

        .info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-bottom: 2rem;
        }

        .info-item {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 1.5rem;
            border-radius: 16px;
            text-align: center;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            border: 1px solid rgba(0, 0, 0, 0.05);
        }

        .info-item .label {
            font-size: 0.9rem;
            color: #6c757d;
            margin-bottom: 0.5rem;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .info-item .value {
            font-weight: 700;
            color: #2c3e50;
            font-size: 1.1rem;
        }

        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 1rem 2rem;
            border-radius: 16px;
            cursor: pointer;
            font-size: 1rem;
            font-weight: 600;
            transition: all 0.3s ease;
            width: 100%;
            margin-bottom: 1rem;
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }

        .btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 12px 35px rgba(102, 126, 234, 0.4);
        }

        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .toggle-buttons {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
        }

        .toggle-btn {
            flex: 1;
            padding: 1rem;
            border: 2px solid #e1e5e9;
            background: white;
            border-radius: 16px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 600;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }

        .toggle-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        .toggle-btn.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: #667eea;
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }

        .loading {
            text-align: center;
            padding: 2rem;
            color: #666;
        }

        .spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        @media (max-width: 768px) {
            .container {
                flex-direction: column;
                padding: 1rem;
                gap: 1rem;
            }
            
            .sidebar {
                width: 100%;
                order: 2;
                padding: 1.5rem;
            }
            
            .map-container {
                order: 1;
                height: 500px;
            }
            
            #map {
                height: 500px;
            }

            .header {
                padding: 1.5rem;
            }

            .header h1 {
                font-size: 2rem;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>📍 Dima M3ak</h1>
        <p>Suivi de Localisation en Direct</p>
    </div>

    <div class="container">
        <div class="sidebar">
            <div id="status" class="status" style="display: none;"></div>

            <div id="sessionInfo" style="display: none;">
                <div class="info-grid">
                    <div class="info-item">
                        <div class="label">Statut</div>
                        <div class="value" id="sessionStatus">-</div>
                    </div>
                    <div class="info-item">
                        <div class="label">Batterie</div>
                        <div class="value" id="batteryLevel">-</div>
                    </div>
                    <div class="info-item">
                        <div class="label">Dernière MAJ</div>
                        <div class="value" id="lastUpdate">-</div>
                    </div>
                    <div class="info-item">
                        <div class="label">Expire</div>
                        <div class="value" id="expiresAt">-</div>
                    </div>
                </div>
            </div>

            <button class="btn" onclick="toggleTracking()" id="trackBtn">Arrêter le Suivi</button>

            <div style="margin-top: 2rem; padding-top: 2rem; border-top: 2px solid #e1e5e9;">
                <h3 style="margin-bottom: 1.5rem; color: #333; font-size: 1.2rem; font-weight: 600;">Affichage de l'Historique</h3>
                <div class="toggle-buttons">
                    <button class="toggle-btn active" onclick="toggleHistory('show')">Afficher le Parcours</button>
                    <button class="toggle-btn" onclick="toggleHistory('hide')">Masquer le Parcours</button>
                </div>
            </div>
        </div>

        <div class="map-container">
            <div id="map"></div>
        </div>
    </div>

    <script>
        let map;
        let currentMarker;
        let historyPolyline;
        let trackingInterval;
        let historyRefreshInterval;
        let isTracking = false;

        // Configuration from Laravel
        const config = {
            dataUrl: '{{ $dataUrl }}',
            historyUrl: '{{ $historyUrl }}',
            publicId: '{{ $publicId }}',
            viewerToken: '{{ $viewerToken }}'
        };

        // Initialize Google Maps
        function initMap() {
            map = new google.maps.Map(document.getElementById('map'), {
                zoom: 15,
                center: { lat: 40.7128, lng: -74.0060 }, // Default to NYC
                mapTypeId: 'roadmap',
                styles: [
                    {
                        featureType: 'poi',
                        elementType: 'labels',
                        stylers: [{ visibility: 'off' }]
                    }
                ]
            });

            // Load initial data and start tracking automatically
            loadInitialData();
        }

        // Load initial session data
        async function loadInitialData() {
            try {
                const response = await fetch(config.dataUrl, {
                    headers: {
                        'Accept': 'application/json',
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                });
                const data = await response.json();

                if (data.error) {
                    showStatus(`Error: ${data.error}`, 'error');
                    return;
                }

                updateSessionInfo(data);
                showStatus('Session chargée avec succès', 'active');
                
                // Center map on current location if available
                if (data.status === 'active' && data.lastLocation) {
                    const position = {
                        lat: parseFloat(data.lastLocation.lat),
                        lng: parseFloat(data.lastLocation.lng)
                    };
                    
                    map.setCenter(position);
                    updateCurrentLocation(position, data.battery);
                }

                // Automatically start tracking and load history
                startTracking();
                loadHistory();
                startHistoryRefresh();

            } catch (error) {
                showStatus(`Échec du chargement de la session: ${error.message}`, 'error');
            }
        }

        // Start live tracking
        function startTracking() {
            if (isTracking) return;

            isTracking = true;
            document.getElementById('trackBtn').textContent = 'Arrêter le Suivi';
            document.getElementById('trackBtn').disabled = false;
            
            showStatus('Suivi en direct démarré', 'active');

            // Update immediately
            updateLocation();

            // Update every 5 seconds
            trackingInterval = setInterval(updateLocation, 5000);
        }

        // Toggle tracking on/off
        function toggleTracking() {
            if (isTracking) {
                stopTracking();
            } else {
                startTracking();
                startHistoryRefresh();
            }
        }

        // Update current location
        async function updateLocation() {
            try {
                const response = await fetch(config.dataUrl, {
                    headers: {
                        'Accept': 'application/json',
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                });
                const data = await response.json();

                if (data.error) {
                    showStatus(`Error: ${data.error}`, 'error');
                    return;
                }

                updateSessionInfo(data);

                if (data.status === 'active' && data.lastLocation) {
                    const position = {
                        lat: parseFloat(data.lastLocation.lat),
                        lng: parseFloat(data.lastLocation.lng)
                    };
                    
                    updateCurrentLocation(position, data.battery);
                } else if (data.status === 'ended') {
                    showStatus('Session has ended', 'ended');
                    stopTracking();
                }

            } catch (error) {
                console.error('Failed to update location:', error);
            }
        }

        // Stop tracking
        function stopTracking() {
            isTracking = false;
            document.getElementById('trackBtn').textContent = 'Démarrer le Suivi';
            document.getElementById('trackBtn').disabled = false;
            
            if (trackingInterval) {
                clearInterval(trackingInterval);
                trackingInterval = null;
            }
            
            if (historyRefreshInterval) {
                clearInterval(historyRefreshInterval);
                historyRefreshInterval = null;
            }
            
            showStatus('Suivi en direct arrêté', 'ended');
        }

        // Update current location marker
        function updateCurrentLocation(position, battery) {
            if (currentMarker) {
                currentMarker.setMap(null);
            }

            // Create custom marker with battery info
            const markerIcon = {
                path: google.maps.SymbolPath.CIRCLE,
                scale: 15,
                fillColor: battery > 20 ? '#4CAF50' : '#F44336',
                fillOpacity: 0.8,
                strokeColor: '#FFFFFF',
                strokeWeight: 3
            };

            currentMarker = new google.maps.Marker({
                position: position,
                map: map,
                icon: markerIcon,
                title: `Current Location (Battery: ${battery}%)`,
                animation: google.maps.Animation.DROP
            });

            // Add info window
            const infoWindow = new google.maps.InfoWindow({
                content: `
                    <div style="padding: 15px; font-family: 'Inter', sans-serif;">
                        <h3 style="margin-bottom: 10px; color: #2c3e50;">📍 Position Actuelle</h3>
                        <p style="margin: 5px 0;"><strong>Batterie:</strong> ${battery}%</p>
                        <p style="margin: 5px 0;"><strong>Coordonnées:</strong> ${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}</p>
                        <p style="margin: 5px 0;"><strong>Heure:</strong> ${new Date().toLocaleTimeString('fr-FR')}</p>
                    </div>
                `
            });

            currentMarker.addListener('click', () => {
                infoWindow.open(map, currentMarker);
            });
        }

        // Load and display history
        async function loadHistory() {
            try {
                const response = await fetch(config.historyUrl);
                const data = await response.json();

                if (data.error) {
                    showStatus(`Erreur lors du chargement de l'historique: ${data.error}`, 'error');
                    return;
                }

                if (data.points && data.points.length > 0) {
                    displayHistoryPath(data.points);
                    if (isTracking) {
                        showStatus(`${data.points.length} points d'historique chargés - Suivi actif`, 'active');
                    }
                } else {
                    if (isTracking) {
                        showStatus('Aucun historique disponible - Suivi actif', 'active');
                    }
                }

            } catch (error) {
                showStatus(`Échec du chargement de l'historique: ${error.message}`, 'error');
            }
        }

        // Start automatic history refresh
        function startHistoryRefresh() {
            // Refresh history every 30 seconds
            historyRefreshInterval = setInterval(loadHistory, 30000);
        }

        // Display history path on map
        function displayHistoryPath(points) {
            // Remove existing polyline
            if (historyPolyline) {
                historyPolyline.setMap(null);
            }

            // Convert points to Google Maps format
            const path = points.map(point => ({
                lat: parseFloat(point.lat),
                lng: parseFloat(point.lng)
            }));

            // Create polyline
            historyPolyline = new google.maps.Polyline({
                path: path,
                geodesic: true,
                strokeColor: '#FF6B6B',
                strokeOpacity: 0.8,
                strokeWeight: 4
            });

            historyPolyline.setMap(map);

            // Only fit bounds if this is the initial load (not a refresh)
            if (!historyRefreshInterval || historyRefreshInterval === null) {
                const bounds = new google.maps.LatLngBounds();
                path.forEach(point => bounds.extend(point));
                map.fitBounds(bounds);
            }
        }

        // Toggle history display
        function toggleHistory(action) {
            const buttons = document.querySelectorAll('.toggle-btn');
            buttons.forEach(btn => btn.classList.remove('active'));
            
            if (action === 'show') {
                document.querySelector('.toggle-btn:first-child').classList.add('active');
                if (historyPolyline) {
                    historyPolyline.setMap(map);
                }
            } else {
                document.querySelector('.toggle-btn:last-child').classList.add('active');
                if (historyPolyline) {
                    historyPolyline.setMap(null);
                }
            }
        }

        // Update session info display
        function updateSessionInfo(data) {
            document.getElementById('sessionInfo').style.display = 'block';
            document.getElementById('sessionStatus').textContent = data.status === 'active' ? 'Actif' : 'Terminé';
            document.getElementById('batteryLevel').textContent = data.battery ? `${data.battery}%` : '-';
            document.getElementById('lastUpdate').textContent = data.lastUpdated ? 
                new Date(data.lastUpdated).toLocaleTimeString('fr-FR') : '-';
            document.getElementById('expiresAt').textContent = data.expiresAt ? 
                new Date(data.expiresAt).toLocaleString('fr-FR') : '-';
        }

        // Show status message
        function showStatus(message, type) {
            const statusEl = document.getElementById('status');
            statusEl.textContent = message;
            statusEl.className = `status ${type}`;
            statusEl.style.display = 'block';
        }
    </script>

    <!-- Google Maps API -->
    <script async defer 
        src="https://maps.googleapis.com/maps/api/js?key=AIzaSyCsltvcUQZtFepHI2tgdRWsn2c7OwOOIm0&callback=initMap">
    </script>
</body>
</html>
